﻿<#
.SYNOPSIS
	This script performs the deployment of a package.
.DESCRIPTION

.PARAMETER DeploymentType
	[PSADT] The type of deployment to perform. Default is: Install.
.PARAMETER DeployMode
	[PSADT] Specifies whether the installation should be run in Interactive, Silent, or NonInteractive mode. Default is: Interactive. Options: Interactive = Shows dialogs, Silent = No dialogs, NonInteractive = Very silent, i.e. no blocking apps. NonInteractive mode is automatically set if it is detected that the process is not user interactive.
.PARAMETER AllowRebootPassThru
	[PSADT] Allows the 3010 return code (requires restart) to be passed back to the parent process (e.g. SCCM) if detected from an installation. If 3010 is passed back to SCCM, a reboot prompt will be triggered.
.PARAMETER TerminalServerMode
	[PSADT] Changes to "user install mode" and back to "user execute mode" for installing/uninstalling applications for Remote Destkop Session Hosts/Citrix servers.
.PARAMETER DisableLogging
	[PSADT] Disables logging to file for the script. Default is: $false.
.PARAMETER InstallMode
	The type of installation to perform. Default is: Install.
	Derived from NI-InstallModes: imUserPart, imWkStaPart, imAppRepair, imReinstall, imModify, imUpdate, imUninstall
.PARAMETER PackagePath
	The path to the package directory or the path to the package information file. Default is: script directory.
.PARAMETER LogFileName
	The name of the log file.
.PARAMETER LogDirectory
	The directory of the log files.
.PARAMETER UserPartHandler
	User-Part handler. Default is: ActiveSetup.
.PARAMETER PersistCache
	Create a package cache entry from the content of the package.
.PARAMETER RunInInteractiveSession
	Restart process to run in interactive session. <number> = specified session; <process name> = session of the specified process; empty, -1 = current user/console session
.PARAMETER RestartInteractive
	Switch to restart process in interactive session - is equivalent to '-RunInInteractiveSession -1'
.PARAMETER InstallationParameters
	User defined installation parameters - format: <name>=<value>[, ...]. In Script.ps1 available as ${Installationparameters.<name>}
.PARAMETER IncludeForUserPart
	List of parameter names to include into the command line for immediate user part execution via separate process and into the command line for Active Setup. Default is: InstallationParameters.
.PARAMETER ExcludeForUserPart
	List of parameter names to exclude from the command line for immediate user part execution via separate process and from the command line for Active Setup. Default is: InstallMode, UserPartHandler, RunInInteractiveSession, RestartInteractive, IncludeForUserPart, ExcludeForUserPart.
.EXAMPLE
    powershell.exe -Command "& { & '.\Deploy-Package.ps1' -DeployMode 'Silent'; Exit $LastExitCode }"
.EXAMPLE
    powershell.exe -Command "& { & '.\Deploy-Package.ps1' -AllowRebootPassThru; Exit $LastExitCode }"
.EXAMPLE
    powershell.exe -Command "& { & '.\Deploy-Package.ps1' -DeploymentType 'Uninstall'; Exit $LastExitCode }"
.EXAMPLE
    Deploy-Application.exe -DeploymentType "Install" -DeployMode "Silent"
.NOTES
	Toolkit Exit Code Ranges:
	60000 - 68999: Reserved for built-in exit codes in Deploy-Application.ps1, Deploy-Application.exe, and AppDeployToolkitMain.ps1
	69000 - 69999: Recommended for user customized exit codes in Deploy-Application.ps1
	70000 - 79999: Recommended for user customized exit codes in AppDeployToolkitExtensions.ps1
.LINK 
	https://www.cancom.de
#>
[CmdletBinding()]
param (	
	# PSADT:
	
	[Parameter(Mandatory=$false)][ValidateSet('Install','Uninstall','Repair')]
	[string]$DeploymentType = 'Install',
	
	[Parameter(Mandatory=$false)][ValidateSet('Interactive','Silent','NonInteractive')]
	[string]$DeployMode = 'Interactive',
	
	[Parameter(Mandatory=$false)][switch]$AllowRebootPassThru = $false,
    [Parameter(Mandatory=$false)][switch]$SuppressRebootPassThru = $false,
	[Parameter(Mandatory=$false)][switch]$TerminalServerMode = $false,
	[Parameter(Mandatory=$false)][switch]$DisableLogging = $false,
	
	# Deploy-Package:
	
	[Parameter(Mandatory=$false)][ValidateSet('InstallUserPart','InstallComputerPart','Install','Repair','Reinstall','Modify','Update','Uninstall')]
	[string]$InstallMode = 'Install',

	[Parameter(Mandatory=$false)][string]$PackagePath = $null,
	[Parameter(Mandatory=$false)][string]$LogFileName = $null,
	[Parameter(Mandatory=$false)][string]$LogDirectory = $null,
	[Parameter(Mandatory=$false)][string]$UserPartHandler = 'ActiveSetup',
	[Parameter(Mandatory=$false)][switch]$PersistCache = $false,

	[Parameter(Mandatory=$false)][Alias("RunInSession")][string]$RunInInteractiveSession = $null,
	[Parameter(Mandatory=$false)][switch]$RestartInteractive = $false,
	[Parameter(Mandatory=$false)][string[]]$InstallationParameters = @(),
	[Parameter(Mandatory=$false)][string[]]$IncludeForUserPart = @('InstallationParameters'),
	[Parameter(Mandatory=$false)][string[]]$ExcludeForUserPart = @('InstallMode','UserPartHandler','RunInInteractiveSession','RestartInteractive','IncludeForUserPart','ExcludeForUserPart')
)

## throw Write-Error etc.
$ErrorActionPreference = "Stop"

function Get-LocalText([string]$localizedString)
{
	$text = $localizedString.Trim();

	$matches = [regex]::Matches($localizedString, "\{(?<culture>\w+)\}\s+(?<text>[^\{]+)");
	if ($matches.Count -gt 0)
	{
		$found = @($matches | where { $_.Groups["culture"].Value -eq [cultureinfo]::CurrentUICulture.TwoLetterISOLanguageName });
		if ($found.Count -eq 0) { $found = @($matches | where { $_.Groups["culture"].Value -eq "en" }); }
		$match = $(if ($found.Count -gt 0) { $found[0] } else { $matches[0] });

		$text = $match.Groups["text"].Value.Trim();
	}

	return $text;
}

function ConvertTo-Boolean([string]$value)
{
	[bool]$flag = 0;
	if ([bool]::TryParse($value, [ref]$flag)) { return $flag; }

	[int]$number = 0;
	if ([int]::TryParse($value, [ref]$number)) { return [bool]$number; }

	return [bool]$value;
}

function ConvertTo-Type([string]$value, [type]$type)
{
	if ([string]::IsNullOrEmpty($value) -or ($type -eq $null)) { return $value; }

	$converter = [System.ComponentModel.TypeDescriptor]::GetConverter($type);
	if (!$converter.CanConvertFrom($value.GetType())) { return $value; }

	return $converter.ConvertFrom($value);
}

function Get-PackageInfo([string]$path)
{
	function Get-PropertyValue($node, [switch]$parameter = $false)
	{
		$properties = $node.SelectNodes("Property");
		if ($properties.Count -gt 0)
		{
			$value = @{}

			foreach ($property in $properties)
			{
				$name = $(if ($property.HasAttribute("Name")) { $property.GetAttribute("Name") } else { $property.Name });
				$value[$name] = (Get-PropertyValue $property -parameter:$parameter);
			}

			return $value;
		}
		
		[string]$value = $node.InnerText;
		if ($parameter)
		{
			if (($node.FirstChild -ne $null) -and ($node.FirstChild.NodeType -eq "Text")) { $value = $node.FirstChild.InnerText; }
			elseif ($node.HasAttribute("Value")) { $value = $node.GetAttribute("Value"); }
			elseif ($node.HasAttribute("DefaultValue")) { $value = $node.GetAttribute("DefaultValue"); }
		}

		$isLocalizable = $(if ($node.HasAttribute("IsLocalizable")) { ConvertTo-Boolean ($node.GetAttribute("IsLocalizable")) } else { $localizedProperties -contains $name });
		if ($isLocalizable) { [string]$value = Get-LocalText $value; }

		$type = $(if ($node.HasAttribute("Type")) { try { Invoke-Expression "[$($node.GetAttribute('Type'))]" } catch { $null } } else { $null });
		if (($type -eq [bool]) -or ($booleanProperties -contains $name)) { [bool]$value = ConvertTo-Boolean $value; }
		elseif ($type -eq [int]) { $number = 0; if ([int]::TryParse($value, [ref]$number)) { [int]$value = $number; } }
		elseif (($type -eq [string[]]) -or ($type -eq [Array])) { [string[]]$value = ([string]$value).Split("`r`n", [System.StringSplitOptions]::RemoveEmptyEntries) }
		elseif ($type -ne $null) { $typedValue = $(try { ConvertTo-Type -value $value -type $type } catch { $value }); [object]$value = $typedValue; };

		return $value;
	}
	
	$path = $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath($path);

	$package = @{};

	$ps1Path = [System.IO.Path]::ChangeExtension($path, ".ps1");
	if ([System.IO.File]::Exists($ps1Path)) { $package = . $ps1Path; }

	$xmlPath = [System.IO.Path]::ChangeExtension($path, ".xml");
	if (![System.IO.File]::Exists($xmlPath)) { return $package; }

	$xml = New-Object System.Xml.XmlDocument;
	$xml.Load($xmlPath);

	$baseProperties = @("ID", "Name", "DisplayName", "Description");
	$node = $xml.DocumentElement.SelectSingleNode("BaseProperties");
	if ($node -ne $null) { $baseProperties = @(([string]$node.InnerText).Split(",") | % { $_.Trim() } | where { ![string]::IsNullOrEmpty($_) }); }

	$localizedProperties = @("DisplayName", "Description");
	$booleanProperties = @("HasUserPart");

	$nodes = @($baseProperties | % { $xml.DocumentElement.SelectSingleNode($_) } | where { $_ -ne $null });
	$nodes += $xml.DocumentElement.SelectNodes("Properties/Property");

	foreach ($node in $nodes)
	{
		$name = $(if ($node.HasAttribute("Name")) { $node.GetAttribute("Name") } else { $node.Name });

		$package[$name] = (Get-PropertyValue $node);
	}

	$parameters = @{}
	$package["Parameters"] = $parameters;

	$nodes = $xml.DocumentElement.SelectNodes("Parameters/Parameter");

	foreach ($node in $nodes)
	{
		$name = $(if ($node.HasAttribute("Name")) { $node.GetAttribute("Name") } else { $node.Name });

		$parameters[$name] = (Get-PropertyValue $node -parameter);
	}

	$xmlDefaultsPath = [System.IO.Path]::Combine([System.IO.Path]::GetDirectoryName($path), "Package.Defaults.xml");
	if (![System.IO.File]::Exists($xmlDefaultsPath)) { return $package; }

	$xmlDefaults = New-Object System.Xml.XmlDocument;
	$xmlDefaults.Load($xmlDefaultsPath);

	$nodes = @($baseProperties | % { $xmlDefaults.DocumentElement.SelectSingleNode($_) } | where { $_ -ne $null });
	$nodes += $xmlDefaults.DocumentElement.SelectNodes("Properties/Property");

	foreach ($node in $nodes)
	{
		$name = $(if ($node.HasAttribute("Name")) { $node.GetAttribute("Name") } else { $node.Name });
		if ($package.ContainsKey($name)) { continue; }

		$package[$name] = (Get-PropertyValue $node);
	}

	$nodes = $xmlDefaults.DocumentElement.SelectNodes("Parameters/Parameter");

	foreach ($node in $nodes)
	{
		$name = $(if ($node.HasAttribute("Name")) { $node.GetAttribute("Name") } else { $node.Name });
		if ($parameters.ContainsKey($name)) { continue; }

		$parameters[$name] = (Get-PropertyValue $node -parameter);
	}

	$internalDefaults = @{ "DisplayVersion" = "Version"; };
	$internalDefaults.GetEnumerator() | where { $package.ContainsKey($_.Key) -and [string]::IsNullOrEmpty($package[$_.Key]) -and $package.ContainsKey($_.Value) } | % { $package[$_.Key] = $package[$_.Value]; }

	return $package;
}

function Merge-ParameterValues($package, [string[]]$parameterValues = @())
{
	if (!($package -is [hashtable])) { return; }

	$parameters = $package["Parameters"];
	if ($parameters -eq $null) { return; }

	foreach ($item in $parameterValues)
	{
		if ([string]::IsNullOrWhiteSpace($item)) { continue; }
		
		$index = $item.IndexOfAny(":="); # <name>=<value> or <name>:<value>
		$name = $(if ($index -ge 0) { $item.Substring(0, $index).Trim() } else { $item.Trim() });
		$value = $(if ($index -ge 0) { $item.Substring($index + 1) } else { $null });
		if (!$parameters.ContainsKey($name)) { continue; }

		$match = [regex]::Match($value, "^b(ase)?64[:=]\s*(?<value>[^\s]*)\s*`$", "IgnoreCase");
		if ($match.Success) { $value = $(try { [System.Text.Encoding]::UTF8.GetString([Convert]::FromBase64String($match.Groups["value"].Value)) } catch { $value }); }

		$parameters[$name] = $value;
	}
}

function ConvertTo-ArgumentString([hashtable]$parameters, [string[]]$include = @(), [string[]]$exclude = @(), [switch]$forScript)
{
	$forExe = !$forScript;

	$items = @();

	foreach ($kvp in $parameters.GetEnumerator())
	{
		$name = $kvp.Key;
		$value = $kvp.Value;

		if (($include.Count -gt 0) -and ($include -notcontains $name)) { continue; }
		if (($exclude.Count -gt 0) -and ($exclude -contains $name)) { continue; }

		$item = $null;

		if ($value -eq $null) { $item = "-$($name):`$null"; }
		elseif ($value -is [switch]) { $item = $(if ($value) { "-$($name)" } else { "-$($name):`$false" }); }
		elseif ($value -is [bool]) { $item = "-$($name) $(if ($value) { 1 } else { 0 })"; }
		elseif ($value -is [datetime]) { $item = "-$($name) `"$($value.ToUniversalTime().ToString('u'))`""; }
		elseif (($value -is [array]) -and $forExe) { $list = @($value | % { $text = $_.ToString([cultureinfo]::InvariantCulture); $(if ($text.IndexOf(" ") -ge 0) { "'$($text)'"; } else { $text }); }); $text = [string]::Join(', ', $list); $item = $(if ($list.Count -eq 0) { "" } elseif ($list.Count -eq 1) { "-$($name) $($text)" } else { "-$($name) `"$($text)`"" }); }
		elseif (($value -is [array]) -and $forScript) { $list = @($value | % { $text = $_.ToString([cultureinfo]::InvariantCulture); $(if ($text.IndexOf(" ") -ge 0) { "'$($text)'"; } else { $text }); }); $text = [string]::Join(', ', $list); $item = $(if ($list.Count -eq 0) { "-$($name) @()" } else { "-$($name) $($text)" }); }
		else { $text = $value.ToString([cultureinfo]::InvariantCulture); if ($text.IndexOf(" ") -ge 0) { $text = "`"$($text)`""; }; $item = "-$($name) $($text)"; }

		if ([string]::IsNullOrEmpty($item)) { continue; }

		$items += $item;
	}
	
	return [string]::Join(" ", $items);
}

try {	
	Push-Location;

	## Set ExecutionPolicy ByPass for process
	try { Set-ExecutionPolicy -ExecutionPolicy ByPass -Scope Process -Force -ErrorAction Stop } catch {}
	
	##*===============================================
	##* VARIABLE DECLARATION
	##*===============================================
	$context = @{

		## Variables: Install Titles (Only set here to override defaults set by the toolkit)
		installName = ''
		installTitle = ''
		
		## Variables: Script
		deployAppScriptFriendlyName = 'Deploy Package'
		deployAppScriptVersion = [version]'1.0.0'
		deployAppScriptDate = ''
		deployAppScriptParameters = $psBoundParameters
		
		## PSADT: exported globals - e.g.: 'env*','appDeploy*'
		adtVariables = $null
		adtFunctions = $null
	}
		
	## Variables: Exit Code
	[int32]$mainExitCode = 0

	if ($PSBoundParameters.ContainsKey("SuppressRebootPassThru")) { $PSBoundParameters["AllowRebootPassThru"] = -not $SuppressRebootPassThru; }
	elseif ($PSBoundParameters.ContainsKey("AllowRebootPassThru")) { $PSBoundParameters["SuppressRebootPassThru"] = -not $AllowRebootPassThru; }

	## Parameters
	if ($PSBoundParameters.ContainsKey("InstallMode"))
	{
		if ($InstallMode -eq "Uninstall") { $DeploymentType = "Uninstall" }
		elseif ($InstallMode -eq "Repair") { $DeploymentType = "Repair" }
		elseif (($InstallMode -eq "InstallUserPart") -and ($UserPartHandler -match "^(ActiveSetup|InternalSetup)\.Uninstall`$")) { $DeploymentType = "Uninstall" }
		elseif (($InstallMode -eq "InstallUserPart") -and ($UserPartHandler -match "^(ActiveSetup|InternalSetup)\.Repair`$")) { $DeploymentType = "Repair" }
		else { $DeploymentType = "Install" }
	}
	elseif ($PSBoundParameters.ContainsKey("DeploymentType"))
	{
		$InstallMode = $DeploymentType;
	}
	
	$context.DeploymentType = $DeploymentType;
	$context.DeployMode = $DeployMode;
	$context.AllowRebootPassThru = $AllowRebootPassThru;
	$context.TerminalServerMode = $TerminalServerMode;
	$context.DisableLogging = $DisableLogging;
	$context.InstallMode = $InstallMode;
	$context.ScriptPath = $null;
	$context.ContinueScript = $null;
	$context.LogFileName = $LogFileName;
	$context.LogDirectory = $LogDirectory;
	$context.UserPartHandler = $UserPartHandler;

	Remove-Variable LogDirectory; # parameter
	
	## Variables: Environment
	If (Test-Path -LiteralPath 'variable:HostInvocation') { $InvocationInfo = $HostInvocation } Else { $InvocationInfo = $MyInvocation }
	[string]$scriptDirectory = Split-Path -Path $InvocationInfo.MyCommand.Definition -Parent
	[string]$scriptName = Split-Path -Path $InvocationInfo.MyCommand.Definition -Leaf
	
	## Context: Package Information
	$packageDirectory = $null;
	$packageInfoPath = $null;
	if ([string]::IsNullOrEmpty($PackagePath)) { $PackagePath = $scriptDirectory; } else { $PackagePath = (Resolve-Path $PackagePath).ProviderPath; }
	if (Test-Path $PackagePath -PathType Container)
	{
		$packageDirectory = $PackagePath;
		$packageInfoPath = Join-Path -Path $packageDirectory -ChildPath Package.xml;
	}
	else # (Test-Path $PackagePath -PathType Leaf)
	{
		$packageInfoPath = $PackagePath;
		$packageDirectory = Split-Path -Path $packageInfoPath -Parent;
	}
	
	$package = Get-PackageInfo -path $packageInfoPath;
	Merge-ParameterValues -package $package -parameterValues $InstallationParameters;

	$context.PackageDirectory = $packageDirectory;
	$context.Package = $package;

	$context.InstallTitle = "$($package.Vendor) $($package.Name) $(if (![string]::IsNullOrEmpty($package.Version) -and (([string]$package.Name).IndexOf($package.Version) -lt 0)) {$package.Version})".Trim();
	
	${LogSource} = "$($scriptName)/$($InstallMode)";
	if (![String]::IsNullOrEmpty($package.Name) -and [String]::IsNullOrEmpty($context.LogFileName))
	{
		$timestamp = [DateTime]::Now.ToString("yyyy-MM-dd-HH-mm-ss-fff");
		$format = $(if (![String]::IsNullOrEmpty($package.LogFileNameFormat)) { $package.LogFileNameFormat }  else { '$($package.Vendor)_$($package.Name)_$($package.Version)_$($InstallMode)_$($timestamp).log' });
		$context.LogFileName = Invoke-Expression "`"$($format)`"";
	}
	
	## Import the PackageDeployment module
	$PackageDeploymentModule = "PackageDeployment";
	try
	{
		if (Test-Path -PathType Container -Path "$($scriptDirectory)\PackageDeployment") { $PackageDeploymentModule = "$($scriptDirectory)\PackageDeployment"; }
		elseif (Test-Path -PathType Leaf -Path "$($scriptDirectory)\PackageDeployment.psd1") { $PackageDeploymentModule = "$($scriptDirectory)\PackageDeployment.psd1"; }
		Import-Module $PackageDeploymentModule -Force -ArgumentList $context;
	}
	catch
	{
		If ($mainExitCode -eq 0){ [int32]$mainExitCode = 60008 }
		Write-Error -Message "Module [$PackageDeploymentModule] failed to load: `n$($_.Exception.Message)`n `n$($_.InvocationInfo.PositionMessage)" -ErrorAction 'Continue'
		## Exit the script, returning the exit code to SCCM
		If (Test-Path -LiteralPath 'variable:HostInvocation') { $script:ExitCode = $mainExitCode; Exit } Else { Exit $mainExitCode }
	}

	## restart interactive
	$restartedProcess = $null;
	if ($RestartInteractive -or $PSBoundParameters.ContainsKey("RunInInteractiveSession"))
	{
		if ($RestartInteractive) { Write-Log -Message "Parameter -RestartInteractive is set, -RunInInteractiveSession is '$($RunInInteractiveSession)'." -Source ${LogSource} }
		else { Write-Log -Message "Parameter -RunInInteractiveSession is set to '$($RunInInteractiveSession)'." -Source ${LogSource} }

		$currentProcess = [System.Diagnostics.Process]::GetCurrentProcess();
		Write-Log -Message "Current process: '$($currentProcess.ProcessName)' with ID $($currentProcess.Id) in session $($currentProcess.SessionId)." -Source ${LogSource}

		$PSPD_RestartedFrom = "PSPD_RestartedFrom";
		$restartedFromProcessID = [System.Environment]::GetEnvironmentVariable($PSPD_RestartedFrom);
		if (![string]::IsNullOrEmpty($restartedFromProcessID))
		{
			Write-Log -Message "Continue in current process: detected environment variable '$($PSPD_RestartedFrom)' (current process was restarted from process with ID $($restartedFromProcessID))." -Source ${LogSource}
		}
		elseif ([PSPD.API]::GetActiveSessionIDs() -contains $currentProcess.SessionId)
		{
			Write-Log -Message "Continue in current process: already running in interactive session $($currentProcess.SessionId)." -Source ${LogSource}
		}
		elseif (((Get-PdContext).ADT -ne $null) -and ![bool]($package.ForceRestartInteractive))
		{
			Write-Log -Message "Continue in current process: Using PSADT ClientServer commands for interactions (see documentation for more information)." -Source ${LogSource}
		}
		else
		{
			if ($package.ForceRestartInteractive) { Write-Log -Message "Package property 'ForceRestartInteractive' is enabled." -Source ${LogSource} }
			
			try
			{
				Write-Log -Message "Restarting the process in the current interactive user session." -Source ${LogSource}

				Write-Log -Message "Sessions:`r`n$(([PSPD.API]::GetSessionInfos() | ft @{l='Session ID';e={$_.SessionID}}, @{l='WinStation name';e={$_.pWinStationName}}, @{l='Connection state';e={$_.State}} -AutoSize | Out-String).Trim())" -Source ${LogSource}

				[System.Environment]::SetEnvironmentVariable($PSPD_RestartedFrom, $currentProcess.Id);
				$restartedProcess = [PSPD.API]::RestartProcessInteractive($RunInInteractiveSession);
				if ($restartedProcess -ne $null)
				{
					Write-Log -Message "Restarted process: '$($restartedProcess.ProcessName)' with ID $($restartedProcess.Id) in session $($restartedProcess.SessionId)." -Source ${LogSource}
					$restartedProcess.WaitForExit();

					Write-Log -Message "The restarted process '$($restartedProcess.ProcessName)' with ID $($restartedProcess.Id) returned: $($restartedProcess.ExitCode)." -Source ${LogSource}
					return; # -> continue in finally-block
				}
				else
				{
					Write-Log -Message "Continue in current process: restart not required." -Source ${LogSource}
				}
			}
			catch [System.OperationCanceledException]
			{
				Write-Log -Message "Continue in current process: $($_.ToString())" -Source ${LogSource}
			}
			catch
			{
				$failed = "Restart the process in interactive mode";
				Write-Log -Message "$($failed).`r`n$(Resolve-Error $_)" -Severity 3 -Source $LogSource;
				Write-Log -Message "Continue in current process: failed to restart process." -Source ${LogSource}
				# throw $_;
			}
		}
	}

	## Import the ShowMessage module
	$ShowMessageModule = "ShowMessage";
	try
	{
		if (Test-Path -PathType Container -Path "$($scriptDirectory)\SupportFiles") { $ShowMessageModule = "$($scriptDirectory)\SupportFiles\ShowMessage.dll"; }
		Import-Module $ShowMessageModule -Force;
	}
	catch
	{
		Write-Log -Message "Module [$ShowMessageModule] failed to load: `n$($_.Exception.Message)`n `n$($_.InvocationInfo.PositionMessage)" -Severity 2 -Source ${LogSource}
	}
	
	$PdContext = Get-PdContext;
	$PdContext.InstallPhase = "$($scriptName)/$($InstallMode)";
	$PdContext.PackageControlsReboot = $true; # allow the package to reboot the computer

	$executeComputerPart = ($PdContext.InstallMode -ne "InstallUserPart");
	$hasUserPart = [bool]$PdContext.Package.HasUserPart;
	$skipUserPart = [bool]$PdContext.Package.SkipImmediateUserPartExecution;
	$executeUserPart = ($hasUserPart -and !$skipUserPart -and ($PdContext.DeployMode -eq "Interactive") -and ($PdContext.InstallMode -ne "InstallComputerPart"));
    $PdContext.IncludeUserPart = ($executeUserPart -and !$PdContext.SessionZero);
    $script:restartForUserPart = ($executeUserPart -and $PdContext.SessionZero);
	$mode = [regex]::Match([string]$PdContext.UserPartHandler, "^(ActiveSetup|InternalSetup)\.(?<mode>.+)").Groups["mode"].Value;
	if ([string]::IsNullOrEmpty($mode)) { $mode = $PdContext.InstallMode; }
	$PdContext.UserPartInstallMode = $(if (($mode -eq "InstallComputerPart") -or ($mode -eq "InstallUserPart")) { "Install" } else { $mode });
	$hideBalloonNotifications = ($PdContext.ShowBalloonNotifications -and ($PdContext.UserPartHandler -match "^InternalSetup(\.|`$)"));
	if ($hideBalloonNotifications) { $PdContext.ShowBalloonNotifications = $false; }
	$showFinalErrorMessage = [bool]$PdContext.Package.ShowFinalErrorMessage;

	function Approve-OnFinish
	{
		Write-Log -Message "Entering Approve-OnFinish"
		try
		{
			$sessionActive = Test-ADTSessionActive;
			if (!$sessionActive)
			{
				Write-Log -Message "No active PSADT session - exit with error" -Severity 2 -Source $LogSource;
				Exit-Package -Status Failed -Message "No active PSADT session."
				return;
			}

			$session = Get-ADTSession;
			$sessionExitCode = $session.GetExitCode();
			Write-Log -Message "Session exit code is: $($sessionExitCode)" -Source $LogSource;

			$pdc = Get-PdContext;
			if (![string]::IsNullOrEmpty($pdc.Status))
			{
				Write-Log -Message "Exit-Status already set: $($pdc.Status) [$($pdc.StatusMessage)]" -Source $LogSource;
				return;
			}

			$cfg = Get-ADTConfig;
			if (($sessionExitCode -eq $cfg.UI.DeferExitCode) -or ($sessionExitCode -eq 60012) -or ($sessionExitCode -eq 1602))
			{
				Write-Log "Exit code $sessionExitCode is defer exit code."
				Exit-Package -Status Undone -ExitCode $sessionExitCode -Message "The package installation has been deferred."
			}
		}
		catch
		{
			Write-Log -Message "$($failed).`r`n$(Resolve-Error $_)" -Severity 3 -Source $LogSource;
		}
		finally
		{
			Write-Log -Message "Leaving Approve-OnFinish."
		}
	}

	Add-ADTModuleCallback -Hookpoint OnFinish -Callback (Get-Item Function:Approve-OnFinish)
	
	# starting PSADT 4 session
	$adtSession = @{
		# App variables.
		AppVendor = $PdContext.Package.Vendor;
		AppName = $PdContext.Package.Name;
		AppVersion = $PdContext.Package.Version;
		AppArch = $PdContext.Package.Architecture;
		AppLang = $PdContext.Package.Language;
		AppRevision = $PdContext.Package.DisplayVersion;
		AppSuccessExitCodes = @(0)
		AppRebootExitCodes = @(1641, 3010)
		AppProcessesToClose = @('regedit', @{ Name = 'msconfig'; Description = 'System Configuration (msconfig)' })
		AppScriptVersion = [version]"$([int]$PdContext.Package.Revision).0";
		AppScriptDate = $PdContext.Package.ScriptDate;
		AppScriptAuthor = $PdContext.Package.ScriptAuthor;
		RequireAdmin = $false;

		# Install Titles (Only set here to override defaults set by the toolkit).
		InstallName = ''
		InstallTitle = ''

		# Script variables.
		DeployAppScriptFriendlyName = $MyInvocation.MyCommand.Name;
		DeployAppScriptParameters = $PSBoundParameters;
		DeployAppScriptVersion = (Get-Module PackageDeployment).Version;
	}
	$adtSession.LogName = $(if (![string]::IsNullOrEmpty($context.LogFileName)) { $context.LogFileName });
	
	# Open a new deployment session, replacing $adtSession with a DeploymentSession.
	if ($PDContext.ADT -ne $null)
	{
		if (![string]::IsNullOrEmpty($context.LogDirectory))
		{
			$adtConfig = (Get-ADTConfig);
			$adtConfig.Toolkit.LogPath = $context.LogDirectory;
			$adtConfig.Toolkit.LogPathNoAdminRights = $context.LogDirectory;
		}
		$iadtParams = Get-ADTBoundParametersAndDefaultValues -Invocation $MyInvocation
		$iadtParams.DeploymentType = $DeploymentType;
		$iadtParams.DeployMode = $DeployMode;
		$adtSession = Remove-ADTHashtableNullOrEmptyValues -Hashtable $adtSession
		Write-Log -Message "Starting ADT session - info [$($adtSession | Out-String)" -Source ${LogSource}
		$adtSession = Open-ADTSession @adtSession @iadtParams -PassThru
		Write-Log -Message "Started ADT session - info [$($adtSession | Out-String)" -Source ${LogSource}
	}
	else { throw "Cannot start ADT session - <PdContext>.ADT is NULL." }

	## Log package deployment information
	$infoSeparator = '#' * 79
	Write-Log -Message $infoSeparator -Source ${LogSource}
	$currentProcess = [System.Diagnostics.Process]::GetCurrentProcess()
	Write-Log -Message "Current process ID: $($currentProcess.Id)" -Source ${LogSource}
	Write-Log -Message "Current process session ID: $($currentProcess.SessionId)" -Source ${LogSource}
	Write-Log -Message "Current process is 64 Bit: $([System.Environment]::Is64BitProcess)" -Source ${LogSource}
	Write-Log -Message "Current process is elevated: $($PdContext.UserIsAdmin)" -Source ${LogSource}
	Write-Log -Message "Executing user account: $($PdContext.User.Process)" -Source ${LogSource}
	Write-Log -Message "Executing user is service account (Context.SessionZero): $($PdContext.SessionZero)" -Source ${LogSource}
	Write-Log -Message "Executing user account: $($PdContext.User.Process)" -Source ${LogSource}
	Write-Log -Message "Logged on user account: $($PdContext.User.LoggedOn)" -Source ${LogSource}
	Write-Log -Message "Package name: $($PdContext.Package.Name)" -Source ${LogSource}
	Write-Log -Message "Package version: $($PdContext.Package.Version)" -Source ${LogSource}
	Write-Log -Message "Package revision: $($PdContext.Package.Revision)" -Source ${LogSource}
	Write-Log -Message "DeployMode: $($PdContext.DeployMode)" -Source ${LogSource}
	Write-Log -Message "DeploymentType: $($PdContext.DeploymentType)" -Source ${LogSource}
	Write-Log -Message "InstallMode: $($PdContext.InstallMode)" -Source ${LogSource}
	Write-Log -Message "Package has user part (Package.HasUserPart): $($hasUserPart)" -Source ${LogSource}
	Write-Log -Message "UserPartHandler: $($PdContext.UserPartHandler)" -Source ${LogSource}
	Write-Log -Message "UserPartInstallMode: $($PdContext.UserPartInstallMode)" -Source ${LogSource}
	Write-Log -Message "Skip immediate user part execution (Package.SkipImmediateUserPartExecution): $($skipUserPart)" -Source ${LogSource}
	Write-Log -Message "Executing computer part: $($executeComputerPart)" -Source ${LogSource}
	$infos = @(if ($PdContext.IncludeUserPart) { if ($executeComputerPart) { "together with computer part" } } elseif ($restartForUserPart) { "via separate process" } else { if ($skipUserPart) { "Package.SkipImmediateUserPartExecution" }; if ($PdContext.DeployMode -ne "Interactive") { "DeployMode $($PdContext.DeployMode)" }; if ($PdContext.InstallMode -eq "InstallComputerPart") { "InstallMode $($PdContext.InstallMode)" } });
	$userPartInfo = $(if ($infos.Count -gt 0) { " ($([string]::Join(', ', $infos)))" } else { $null });
	Write-Log -Message "Executing user part: $($executeUserPart)$($userPartInfo)" -Source ${LogSource}
	if ($hideBalloonNotifications) { Write-Log -Message "Hiding balloon notifications: $(!$PdContext.ShowBalloonNotifications)" -Source ${LogSource}; }
	Write-Log -Message "Installed Apps registration: $($PdContext.InstalledAppsRegistryKey)" -Source ${LogSource}
	Write-Log -Message $infoSeparator -Source ${LogSource}

	## Perform package scripts
	$packageScripts = @();
	if (!$package.ContainsKey("Script") -or [string]::IsNullOrEmpty($package.Script)) { $packageScripts = @("Script.ps1"); }
	elseif ($package.Script -is [string]) { $packageScripts = @($package.Script); }
	elseif ($package.Script -is [string[]]) { $packageScripts = $package.Script; }

	$packageScripts = @($packageScripts | % { [regex]::Replace($_, "(^[\.\\/]+[\\/]|^[\\/]+|[\\/]+\.+(?=[\\/]+))", "") });
	
	$gotoPs1Path = "$($packageDirectory)\goto.ps1";
	
	Write-Log -Message "Processing scripts ($($packageScripts.Count)): $([string]::Join(' ', $packageScripts))." -Source ${LogSource}
	foreach ($script in $packageScripts)
	{
		$PdContext.InstallPhase = "$($script)/$($InstallMode)";
		$scriptPath = "$($packageDirectory)\$($script)"
		if (Test-Path $scriptPath)
		{
			Set-Location $packageDirectory;
			Write-Log -Message "Current location is: '$((Get-Location).Path)'." -Source ${LogSource}
			[System.Environment]::CurrentDirectory = (Get-Location -PSProvider FileSystem).ProviderPath;
			Write-Log -Message "Working directory is: '$([System.Environment]::CurrentDirectory)'." -Source ${LogSource}
			
			$PdContext.ScriptPath = $scriptPath;
			$PdContext.ContinueScript = $null;
			Write-Log -Message "Processing script '$($scriptPath)' for mode '$($InstallMode)'." -Source ${LogSource}
			Clear-DeleteAtEndOfScript;

			. $scriptPath
			
			while ($PdContext.ContinueScript -is [scriptblock])
			{
				$continueScript = $PdContext.ContinueScript;

				if (Test-Path $gotoPs1Path)
				{
					$PdContext.ContinueScript = $continueScript.ToString();

					Write-Log -Message "Continue with goto script ($($gotoPs1Path)) ..." -Source ${LogSource}
					. $gotoPs1Path;
				}
				else
				{
					$PdContext.ContinueScript = $null;
					$tmpPath = [System.IO.Path]::Combine([System.IO.Path]::GetTempPath(), "$([Guid]::NewGuid()).ps1");
					[System.IO.File]::WriteAllText($tmpPath, $continueScript.ToString(), [System.Text.Encoding]::UTF8);
					
					Write-Log -Message "Continue script '$($tmpPath)' ..." -Source ${LogSource}
					. $tmpPath;

					try { [System.IO.File]::Delete($tmpPath); } catch {}
				}
			}

			if ([string]::IsNullOrEmpty($LogName))
			{
				$global:LogName = $LogFileName;
				Write-Log -Message "Script has finished and the log file name is empty. Restored the log file name to '$($LogName)'." -Source ${LogSource};

				if ([string]::IsNullOrEmpty($InstallName))
				{
					$global:InstallName = $package.Name;
					Write-Log -Message "Restored the installation name to '$($InstallName)'." -Source ${LogSource};
				}

				if ($LASTEXITCODE -is [int])
				{
					# e.g. via Show-InstallationPrompt (-ExitOnTimeout), Execute-Process (-ExitOnProcessFailure), Show-InstallationWelcome (timeout, defer, disk space requirements), or caused by an assembly load failure. 
					Write-Log -Message "Maybe there was an explicit call to Exit-Script, which set the last exit code to $($LASTEXITCODE)." -Source ${LogSource};

					if (($mainExitCode -eq 0) -and ($mainExitCode -ne $LASTEXITCODE))
					{
						$mainExitCode = $LASTEXITCODE;
						Write-Log -Message "Setting the main exit code to the last exit code ($($mainExitCode))." -Source ${LogSource};
					}
					elseif ($mainExitCode -ne 0)
					{
						Write-Log -Message "The current main exit code is $($mainExitCode) - ignoring the last exit code ($($LASTEXITCODE))." -Source ${LogSource};
					}
					
					if ($global:configShowBalloonNotifications)
					{
						$global:configShowBalloonNotifications = $false;
						Write-Log -Message "Disabled balloon notifications for the final Exit-Script call." -Source ${LogSource};
					}
				}
			}

			Invoke-DeleteAtEndOfScript
			
			Write-Log -Message "Finished script '$($scriptPath)'. Reboot requests: $($PdContext.RebootRequests)" -Source ${LogSource}
			
			if (![string]::IsNullOrEmpty($PdContext.Status))
			{
				switch ($PdContext.Status)
				{
					"Done" { $mainExitCode = 0; break; }
					"Undone" { $mainExitCode = $([int]$n = 0; if ([int]::TryParse($PdContext.Package.ExitCodeOnPackageStatusUndone, [ref]$n)) {$n} else {69000}); break; }
					"UndoneContinueParentScript" { $mainExitCode = 69001; break; }
					"Failed" { $mainExitCode = 69002; break; }
					default { throw "Unknown exit status '$($PdContext.Status)'."; }
				}

				[string]$statusCode = $PdContext.StatusCode
				$match = [regex]::Match($statusCode, "\d+");
				if ($match.Success)
				{
					$mainExitCode = [int]::Parse($match.Value);
					$PdContext.StatusCode = [string]::Concat($statusCode.Substring(0, $match.Index), $statusCode.Substring($match.Index + $match.Length)).Trim(":= ");
					if ([string]::IsNullOrEmpty($PdContext.StatusCode)) { $PdContext.StatusCode = $mainExitCode }
				}

				$status = "'$($PdContext.Status)'"; if (![string]::IsNullOrEmpty($PdContext.StatusMessage)) { $status += " [$($PdContext.StatusMessage)]"; }
				Write-Log -Message "Exit status is $($status) - setting exit code to $($mainExitCode)." -Source ${LogSource}
				break; # -> exit foreach
			}
			else
			{
				if (($mainExitCode -eq $PdContext.PSADT.InstallationDeferExitCode) -and ![string]::IsNullOrEmpty($PDContext.Package.PdStatusOnInstallationDeferExitCode))
				{
					$PdContext.Status = [string]$PDContext.Package.PdStatusOnInstallationDeferExitCode;
					Write-Log -Message "Exit code is $($mainExitCode) - setting exit status to $($PdContext.Status)." -Source ${LogSource}
				}
				elseif (($mainExitCode -eq $PdContext.PSADT.InstallationUIExitCode) -and ![string]::IsNullOrEmpty($PDContext.Package.PdStatusOnInstallationUIExitCode))
				{
					$PdContext.Status = [string]$PDContext.Package.PdStatusOnInstallationUIExitCode;
					Write-Log -Message "Exit code is $($mainExitCode) - setting exit status to $($PdContext.Status)." -Source ${LogSource}
				}
			}
		}
		else
		{
			throw "Script '$($scriptPath)' for mode '$($InstallMode)' not present."
		}
	}

	# MSI: 3010 = ERROR_SUCCESS_REBOOT_REQUIRED, 1641 = ERROR_SUCCESS_REBOOT_INITIATED
	if (($mainExitCode -eq 0) -and ([int]$PdContext.RebootRequests -gt 0))
	{
		$mainExitCode = 3010;

		if ($PdContext.StartSystemShutdownParameters -ne $null)
		{
			$parameters = $PdContext.StartSystemShutdownParameters;
			$action = $(if ($parameters.ContainsKey("Restart")) { "reboot" } else { "shutdown" });
			Write-Log -Message "The package requested a $($action) of the computer (request count: $($PdContext.RebootRequests))." -Source ${LogSource}

			if ($PdContext.PackageControlsReboot)
			{
				Write-Log -Message "Initializing the requested $($action)." -Source ${LogSource}
				$success = Start-SystemShutdown -PassThru @parameters;
				if ($success) { $mainExitCode = 1641; }
			}
			else
			{
				Write-Log -Message "The package is not allowed to shut down or restart the computer." -Source ${LogSource}
			}
		}
		
		Write-Log -Message "Reboot requests: $($PdContext.RebootRequests) - setting exit code to $($mainExitCode)." -Source ${LogSource}
	}
}
catch
{
	[int32]$mainExitCode = 60001
	[string]$mainErrorMessage = "$(Resolve-Error $_)"
	Write-Log -Message $mainErrorMessage -Severity 3 -Source ${LogSource}

	if ([string]::IsNullOrEmpty($PdContext.Status) -and ![string]::IsNullOrEmpty($PDContext.Package.PdStatusOnException))
	{
		$PdContext.Status = [string]$PDContext.Package.PdStatusOnException;
		$PdContext.StatusMessage = $_.Exception.Message;
		Write-Log -Message "Failed with exception: setting exit status to '$($PdContext.Status)' (PdStatusOnException) and status message to '$($PdContext.StatusMessage)'." -Source ${LogSource}
	}

	if (($DeployMode -eq "Interactive") -and $showFinalErrorMessage) { $void = Show-DialogBox -Text $mainErrorMessage -Icon "Stop" }
}
finally
{
	Pop-Location;

	if ($restartedProcess -ne $null)
	{
		# note: 
		# when finishing the "## restart interactive" sequence
		# some values may be unavailable here (e.g. $PdContext is undefined)

		Write-Log -Message "Main exit code is: $($mainExitCode)." -Source ${LogSource}

		$exitCode = $restartedProcess.ExitCode;
		Write-Log -Message "Returning the exit code from the restarted process '$($restartedProcess.ProcessName)' with ID $($restartedProcess.Id) in session $($restartedProcess.SessionId): $($exitCode)." -Source ${LogSource}
		Exit-Script -ExitCode $exitCode;
	}

	$PdContext.InstallPhase = "$($scriptName)/Finalization";

	Write-InstalledAppsRegistry -ExitCode $mainExitCode -WriteAllProperties -ContinueOnError;
	$success = ($mainExitCode -in @(0, 1641, 3010));
	
	$myCommandPath = $InvocationInfo.MyCommand.Definition;
	if ($success -and $PersistCache)
	{
		$entryDir = New-PackageCacheEntry -ScriptPath $myCommandPath -ContinueOnError;
		if (![string]::IsNullOrEmpty($entryDir)) { Write-Log -Message "Package cache entry created at '$($entryDir)'." -Source ${LogSource}; }
		else { Write-Log -Message "No package cache entry created - although -PersistCache is set." -Severity 2 -Source ${LogSource}; }
		if (![string]::IsNullOrEmpty($myCommandPath))
		{
			if (![string]::IsNullOrEmpty($entryDir)) { $myCommandPath = [System.IO.Path]::Combine($entryDir, [System.IO.Path]::GetFileName($myCommandPath)); }
			Write-Log -Message "Using script path '$($myCommandPath)'." -Source ${LogSource}
		}
	}
	
	if ($success) { Write-ActiveSetupRegistry -ScriptPath $myCommandPath -Arguments (ConvertTo-ArgumentString -parameters $PSBoundParameters -include $IncludeForUserPart -exclude $ExcludeForUserPart) -ContinueOnError; }
	else { Write-Log -Message "Skip writing Active Setup Registry: Exit code is $($mainExitCode)." -Source ${LogSource} }

	if ($success -and $script:restartForUserPart -and ($mainExitCode -eq 1641))
	{
		Write-Log -Message "Skip invoking user part installation: a system reboot was initiated (exit code $($mainExitCode))." -Source ${LogSource}
	}
	elseif ($success -and $script:restartForUserPart)
	{
		Write-Log -Message "Executing user part via separate process." -Source ${LogSource}
		$userPartExitCode = 0;

		try
		{
			$applicationName = [System.IO.Path]::Combine([System.IO.Path]::GetDirectoryName($myCommandPath), "Deploy-Application.exe");
			$optionalArguments = ConvertTo-ArgumentString -parameters $PSBoundParameters -include $IncludeForUserPart -exclude $ExcludeForUserPart;
			$arguments = "-InstallMode InstallUserPart -UserPartHandler InternalSetup.$($PdContext.InstallMode) $($optionalArguments)".Trim();

			$process = $null;
			if ($PdContext.SessionZero)
			{
				Write-Log -Message "Running '$($applicationName)' with arguments [$($arguments)] in logged-on user session." -Source ${LogSource};
				$process = [PSPD.API]::StartInteractive($applicationName, $arguments, $null, "~");
				Write-Log -Message "Started process '$($process.ProcessName)' with ID $($process.Id) in user session $($process.SessionId)." -Source ${LogSource};
			}
			else
			{
				Write-Log -Message "Running '$($applicationName)' with arguments [$($arguments)] in current session." -Source ${LogSource};
				$process = [System.Diagnostics.Process]::Start($applicationName, $arguments);
				Write-Log -Message "Started process '$($process.ProcessName)' with ID $($process.Id) in current session ($($process.SessionId))." -Source ${LogSource};
			}

			$process.WaitForExit();
			$userPartExitCode = $process.ExitCode;
			Write-Log -Message "Process '$($process.ProcessName)' with ID $($process.Id) terminated with exit code $($userPartExitCode)." -Source ${LogSource};
		}
		catch
		{
			$userPartExitCode = 60001
			$userPartErrorMessage = "$($failed).`r`n$(Resolve-Error $_)";
			Write-Log -Message $userPartErrorMessage -Severity 3 -Source ${LogSource}
			if ($DeployMode -eq "Interactive") { $void = Show-DialogBox -Text $userPartErrorMessage -Icon "Stop" }
		}
		
		if ($userPartExitCode -ne 0)
		{
			Write-Log -Message "Inital process exit code: $($mainExitCode)." -Source ${LogSource};
			Write-Log -Message "User part process exit code: $($userPartExitCode)." -Source ${LogSource};

			$mainExitCode = $userPartExitCode;
			Write-Log -Message "Resulting exit code: $($mainExitCode)." -Source ${LogSource};
			$success = ($mainExitCode -in @(0, 1641, 3010));
			Write-Log -Message "Resulting status: $(if ($success) {'success'} else {'failure'})." -Source ${LogSource};
		}
	}

	Exit-Script -ExitCode $mainExitCode
}

# SIG # Begin signature block
# MIImUAYJKoZIhvcNAQcCoIImQTCCJj0CAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCCC9Y6ISDqNuiGE
# Ny7QduvLnoRNcyyk1I8gjGTULkQXXaCCH2UwggYUMIID/KADAgECAhB6I67aU2mW
# D5HIPlz0x+M/MA0GCSqGSIb3DQEBDAUAMFcxCzAJBgNVBAYTAkdCMRgwFgYDVQQK
# Ew9TZWN0aWdvIExpbWl0ZWQxLjAsBgNVBAMTJVNlY3RpZ28gUHVibGljIFRpbWUg
# U3RhbXBpbmcgUm9vdCBSNDYwHhcNMjEwMzIyMDAwMDAwWhcNMzYwMzIxMjM1OTU5
# WjBVMQswCQYDVQQGEwJHQjEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMSwwKgYD
# VQQDEyNTZWN0aWdvIFB1YmxpYyBUaW1lIFN0YW1waW5nIENBIFIzNjCCAaIwDQYJ
# KoZIhvcNAQEBBQADggGPADCCAYoCggGBAM2Y2ENBq26CK+z2M34mNOSJjNPvIhKA
# VD7vJq+MDoGD46IiM+b83+3ecLvBhStSVjeYXIjfa3ajoW3cS3ElcJzkyZlBnwDE
# JuHlzpbN4kMH2qRBVrjrGJgSlzzUqcGQBaCxpectRGhhnOSwcjPMI3G0hedv2eNm
# GiUbD12OeORN0ADzdpsQ4dDi6M4YhoGE9cbY11XxM2AVZn0GiOUC9+XE0wI7CQKf
# OUfigLDn7i/WeyxZ43XLj5GVo7LDBExSLnh+va8WxTlA+uBvq1KO8RSHUQLgzb1g
# bL9Ihgzxmkdp2ZWNuLc+XyEmJNbD2OIIq/fWlwBp6KNL19zpHsODLIsgZ+WZ1AzC
# s1HEK6VWrxmnKyJJg2Lv23DlEdZlQSGdF+z+Gyn9/CRezKe7WNyxRf4e4bwUtrYE
# 2F5Q+05yDD68clwnweckKtxRaF0VzN/w76kOLIaFVhf5sMM/caEZLtOYqYadtn03
# 4ykSFaZuIBU9uCSrKRKTPJhWvXk4CllgrwIDAQABo4IBXDCCAVgwHwYDVR0jBBgw
# FoAU9ndq3T/9ARP/FqFsggIv0Ao9FCUwHQYDVR0OBBYEFF9Y7UwxeqJhQo1SgLqz
# YZcZojKbMA4GA1UdDwEB/wQEAwIBhjASBgNVHRMBAf8ECDAGAQH/AgEAMBMGA1Ud
# JQQMMAoGCCsGAQUFBwMIMBEGA1UdIAQKMAgwBgYEVR0gADBMBgNVHR8ERTBDMEGg
# P6A9hjtodHRwOi8vY3JsLnNlY3RpZ28uY29tL1NlY3RpZ29QdWJsaWNUaW1lU3Rh
# bXBpbmdSb290UjQ2LmNybDB8BggrBgEFBQcBAQRwMG4wRwYIKwYBBQUHMAKGO2h0
# dHA6Ly9jcnQuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY1RpbWVTdGFtcGluZ1Jv
# b3RSNDYucDdjMCMGCCsGAQUFBzABhhdodHRwOi8vb2NzcC5zZWN0aWdvLmNvbTAN
# BgkqhkiG9w0BAQwFAAOCAgEAEtd7IK0ONVgMnoEdJVj9TC1ndK/HYiYh9lVUacah
# RoZ2W2hfiEOyQExnHk1jkvpIJzAMxmEc6ZvIyHI5UkPCbXKspioYMdbOnBWQUn73
# 3qMooBfIghpR/klUqNxx6/fDXqY0hSU1OSkkSivt51UlmJElUICZYBodzD3M/SFj
# eCP59anwxs6hwj1mfvzG+b1coYGnqsSz2wSKr+nDO+Db8qNcTbJZRAiSazr7KyUJ
# Go1c+MScGfG5QHV+bps8BX5Oyv9Ct36Y4Il6ajTqV2ifikkVtB3RNBUgwu/mSiSU
# ice/Jp/q8BMk/gN8+0rNIE+QqU63JoVMCMPY2752LmESsRVVoypJVt8/N3qQ1c6F
# ibbcRabo3azZkcIdWGVSAdoLgAIxEKBeNh9AQO1gQrnh1TA8ldXuJzPSuALOz1Uj
# b0PCyNVkWk7hkhVHfcvBfI8NtgWQupiaAeNHe0pWSGH2opXZYKYG4Lbukg7HpNi/
# KqJhue2Keak6qH9A8CeEOB7Eob0Zf+fU+CCQaL0cJqlmnx9HCDxF+3BLbUufrV64
# EbTI40zqegPZdA+sXCmbcZy6okx/SjwsusWRItFA3DE8MORZeFb6BmzBtqKJ7l93
# 9bbKBy2jvxcJI98Va95Q5JnlKor3m0E7xpMeYRriWklUPsetMSf2NvUQa/E5vVye
# fQIwggYaMIIEAqADAgECAhBiHW0MUgGeO5B5FSCJIRwKMA0GCSqGSIb3DQEBDAUA
# MFYxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxLTArBgNV
# BAMTJFNlY3RpZ28gUHVibGljIENvZGUgU2lnbmluZyBSb290IFI0NjAeFw0yMTAz
# MjIwMDAwMDBaFw0zNjAzMjEyMzU5NTlaMFQxCzAJBgNVBAYTAkdCMRgwFgYDVQQK
# Ew9TZWN0aWdvIExpbWl0ZWQxKzApBgNVBAMTIlNlY3RpZ28gUHVibGljIENvZGUg
# U2lnbmluZyBDQSBSMzYwggGiMA0GCSqGSIb3DQEBAQUAA4IBjwAwggGKAoIBgQCb
# K51T+jU/jmAGQ2rAz/V/9shTUxjIztNsfvxYB5UXeWUzCxEeAEZGbEN4QMgCsJLZ
# UKhWThj/yPqy0iSZhXkZ6Pg2A2NVDgFigOMYzB2OKhdqfWGVoYW3haT29PSTahYk
# wmMv0b/83nbeECbiMXhSOtbam+/36F09fy1tsB8je/RV0mIk8XL/tfCK6cPuYHE2
# 15wzrK0h1SWHTxPbPuYkRdkP05ZwmRmTnAO5/arnY83jeNzhP06ShdnRqtZlV59+
# 8yv+KIhE5ILMqgOZYAENHNX9SJDm+qxp4VqpB3MV/h53yl41aHU5pledi9lCBbH9
# JeIkNFICiVHNkRmq4TpxtwfvjsUedyz8rNyfQJy/aOs5b4s+ac7IH60B+Ja7TVM+
# EKv1WuTGwcLmoU3FpOFMbmPj8pz44MPZ1f9+YEQIQty/NQd/2yGgW+ufflcZ/ZE9
# o1M7a5Jnqf2i2/uMSWymR8r2oQBMdlyh2n5HirY4jKnFH/9gRvd+QOfdRrJZb1sC
# AwEAAaOCAWQwggFgMB8GA1UdIwQYMBaAFDLrkpr/NZZILyhAQnAgNpFcF4XmMB0G
# A1UdDgQWBBQPKssghyi47G9IritUpimqF6TNDDAOBgNVHQ8BAf8EBAMCAYYwEgYD
# VR0TAQH/BAgwBgEB/wIBADATBgNVHSUEDDAKBggrBgEFBQcDAzAbBgNVHSAEFDAS
# MAYGBFUdIAAwCAYGZ4EMAQQBMEsGA1UdHwREMEIwQKA+oDyGOmh0dHA6Ly9jcmwu
# c2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY0NvZGVTaWduaW5nUm9vdFI0Ni5jcmww
# ewYIKwYBBQUHAQEEbzBtMEYGCCsGAQUFBzAChjpodHRwOi8vY3J0LnNlY3RpZ28u
# Y29tL1NlY3RpZ29QdWJsaWNDb2RlU2lnbmluZ1Jvb3RSNDYucDdjMCMGCCsGAQUF
# BzABhhdodHRwOi8vb2NzcC5zZWN0aWdvLmNvbTANBgkqhkiG9w0BAQwFAAOCAgEA
# Bv+C4XdjNm57oRUgmxP/BP6YdURhw1aVcdGRP4Wh60BAscjW4HL9hcpkOTz5jUug
# 2oeunbYAowbFC2AKK+cMcXIBD0ZdOaWTsyNyBBsMLHqafvIhrCymlaS98+QpoBCy
# KppP0OcxYEdU0hpsaqBBIZOtBajjcw5+w/KeFvPYfLF/ldYpmlG+vd0xqlqd099i
# ChnyIMvY5HexjO2AmtsbpVn0OhNcWbWDRF/3sBp6fWXhz7DcML4iTAWS+MVXeNLj
# 1lJziVKEoroGs9Mlizg0bUMbOalOhOfCipnx8CaLZeVme5yELg09Jlo8BMe80jO3
# 7PU8ejfkP9/uPak7VLwELKxAMcJszkyeiaerlphwoKx1uHRzNyE6bxuSKcutisqm
# KL5OTunAvtONEoteSiabkPVSZ2z76mKnzAfZxCl/3dq3dUNw4rg3sTCggkHSRqTq
# lLMS7gjrhTqBmzu1L90Y1KWN/Y5JKdGvspbOrTfOXyXvmPL6E52z1NZJ6ctuMFBQ
# ZH3pwWvqURR8AgQdULUvrxjUYbHHj95Ejza63zdrEcxWLDX6xWls/GDnVNueKjWU
# H3fTv1Y8Wdho698YADR7TNx8X8z2Bev6SivBBOHY+uqiirZtg0y9ShQoPzmCcn63
# Syatatvx157YK9hlcPmVoa1oDE5/L9Uo2bC5a4CH2RwwggY/MIIEp6ADAgECAhA2
# nIGBal4DSIUR8kAAOO1LMA0GCSqGSIb3DQEBDAUAMFQxCzAJBgNVBAYTAkdCMRgw
# FgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxKzApBgNVBAMTIlNlY3RpZ28gUHVibGlj
# IENvZGUgU2lnbmluZyBDQSBSMzYwHhcNMjMxMjAxMDAwMDAwWhcNMjYxMTMwMjM1
# OTU5WjBWMQswCQYDVQQGEwJERTEbMBkGA1UECAwSQmFkZW4tV8O8cnR0ZW1iZXJn
# MRQwEgYDVQQKDAtDQU5DT00gR21iSDEUMBIGA1UEAwwLQ0FOQ09NIEdtYkgwggIi
# MA0GCSqGSIb3DQEBAQUAA4ICDwAwggIKAoICAQDArSw9LTzTaw9/KvfNlE0SFvf9
# qcOQ/lXAPYkhX3xVx2L1/MrBosbFLHvJJ6iWD2z4717VRgN/f4woMx1ZZ+k5so33
# kL6rTKliDv1C91rd1aI4NGxwVMg0wOLKA0UDN0oR8/WgKES3QBJYq//zi3+VvART
# ml1dpOcQje3dXPcRY1m8zo/ObmB8pPr8pQAYTgmZuYtj+IZe1G96HnEMaglZT7Ko
# 9CiuCT6MoUuB7P2w+JMpRRUSW7sEk1eex3+g6iL/NcthpF3YqWCgWwNHys0boYNa
# eQuH2BC4o4Lv2yXdR8nK1SuVaYelCd4b5hICwL3as6+z8Y2J0HEY3focyhtZsHta
# hMk8STDcjvgWeap79Qv5jktC7pPdKxv1aI/U0d0V/vzQv+aJ0ySIrHbGMrwGMlqS
# RULednSIxm0YqUnay3C+w+9g+cF9A3TcAGHJwiBZrAvYexcMz4yC/ZpEIVpdnfzJ
# ZQkzqUO0AuBg0qoXbTGb67ZZwHoJKiZjLXRYuWK3Swzei0wj1VWqvmMgEVC9zAI3
# pC0kYXq9OUvFhEYyQ5iCgbluVK1Q60BwsuP1L1z81PH+0gfPpIln1cg35Dp5xL6O
# +wOC1J5GmlqfAzbwVYblBjAo8J0oZgBGHA1akmFBEINDTHxdDgNh+f/2Jop9z/gm
# 4uXWBzSSRVw6TbCfaQIDAQABo4IBiTCCAYUwHwYDVR0jBBgwFoAUDyrLIIcouOxv
# SK4rVKYpqhekzQwwHQYDVR0OBBYEFDtSj69rtLzxAlC2x8F+gn+1H8w/MA4GA1Ud
# DwEB/wQEAwIHgDAMBgNVHRMBAf8EAjAAMBMGA1UdJQQMMAoGCCsGAQUFBwMDMEoG
# A1UdIARDMEEwNQYMKwYBBAGyMQECAQMCMCUwIwYIKwYBBQUHAgEWF2h0dHBzOi8v
# c2VjdGlnby5jb20vQ1BTMAgGBmeBDAEEATBJBgNVHR8EQjBAMD6gPKA6hjhodHRw
# Oi8vY3JsLnNlY3RpZ28uY29tL1NlY3RpZ29QdWJsaWNDb2RlU2lnbmluZ0NBUjM2
# LmNybDB5BggrBgEFBQcBAQRtMGswRAYIKwYBBQUHMAKGOGh0dHA6Ly9jcnQuc2Vj
# dGlnby5jb20vU2VjdGlnb1B1YmxpY0NvZGVTaWduaW5nQ0FSMzYuY3J0MCMGCCsG
# AQUFBzABhhdodHRwOi8vb2NzcC5zZWN0aWdvLmNvbTANBgkqhkiG9w0BAQwFAAOC
# AYEAJjsMqKSeh1xy940zL3vIr3V1dl2f7Qp49GXD+7pXXfHABvw/y7AZFLCTPoHm
# kV71A3VBbQ9xc2Avzd1lQOXolYC/SuO7CWPPLpcK7j15YxurHKhxufC50f6ECmg9
# ziRMtKIvbjJRX5FDXFfHsEcNJ+3gX2jgaWxB0UgdqE/p4ionURoMze8yHHrQF59b
# r2lAFwr4IQzVjDsh/2Kki8QugddiA506lbDHiVaxKi29hVQ/EXXzWpUzJf8rgCaY
# jKSEXAYq264QDPnEdvq7oTjXG/25VCodN/H7EpHaHv10kSvUIL8QV4ZUKPab631V
# FavfLaKHgQhYjzWdXe71qcvuLdaSObTbzWlwK9/l9G3cj5HYpnyaAba1nhIeMiWi
# iAc0tGEfS9DVfnbdr1zn3U/PiAVbsoaXx4RhlvIBnMOke2HIOcpVgWraHDN1j73v
# yOagdJBbD1alLFuFVoVhMDQ6OHQcSUBt/xuZ/SUsauguKfdmlPfYwGsdA/f1mAks
# 8ZrRMIIGYjCCBMqgAwIBAgIRAKQpO24e3denNAiHrXpOtyQwDQYJKoZIhvcNAQEM
# BQAwVTELMAkGA1UEBhMCR0IxGDAWBgNVBAoTD1NlY3RpZ28gTGltaXRlZDEsMCoG
# A1UEAxMjU2VjdGlnbyBQdWJsaWMgVGltZSBTdGFtcGluZyBDQSBSMzYwHhcNMjUw
# MzI3MDAwMDAwWhcNMzYwMzIxMjM1OTU5WjByMQswCQYDVQQGEwJHQjEXMBUGA1UE
# CBMOV2VzdCBZb3Jrc2hpcmUxGDAWBgNVBAoTD1NlY3RpZ28gTGltaXRlZDEwMC4G
# A1UEAxMnU2VjdGlnbyBQdWJsaWMgVGltZSBTdGFtcGluZyBTaWduZXIgUjM2MIIC
# IjANBgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEA04SV9G6kU3jyPRBLeBIHPNyU
# gVNnYayfsGOyYEXrn3+SkDYTLs1crcw/ol2swE1TzB2aR/5JIjKNf75QBha2Ddj+
# 4NEPKDxHEd4dEn7RTWMcTIfm492TW22I8LfH+A7Ehz0/safc6BbsNBzjHTt7FngN
# fhfJoYOrkugSaT8F0IzUh6VUwoHdYDpiln9dh0n0m545d5A5tJD92iFAIbKHQWGb
# CQNYplqpAFasHBn77OqW37P9BhOASdmjp3IijYiFdcA0WQIe60vzvrk0HG+iVcwV
# Zjz+t5OcXGTcxqOAzk1frDNZ1aw8nFhGEvG0ktJQknnJZE3D40GofV7O8WzgaAnZ
# moUn4PCpvH36vD4XaAF2CjiPsJWiY/j2xLsJuqx3JtuI4akH0MmGzlBUylhXvdNV
# XcjAuIEcEQKtOBR9lU4wXQpISrbOT8ux+96GzBq8TdbhoFcmYaOBZKlwPP7pOp5M
# zx/UMhyBA93PQhiCdPfIVOCINsUY4U23p4KJ3F1HqP3H6Slw3lHACnLilGETXRg5
# X/Fp8G8qlG5Y+M49ZEGUp2bneRLZoyHTyynHvFISpefhBCV0KdRZHPcuSL5OAGWn
# BjAlRtHvsMBrI3AAA0Tu1oGvPa/4yeeiAyu+9y3SLC98gDVbySnXnkujjhIh+oaa
# tsk/oyf5R2vcxHahajMCAwEAAaOCAY4wggGKMB8GA1UdIwQYMBaAFF9Y7UwxeqJh
# Qo1SgLqzYZcZojKbMB0GA1UdDgQWBBSIYYyhKjdkgShgoZsx0Iz9LALOTzAOBgNV
# HQ8BAf8EBAMCBsAwDAYDVR0TAQH/BAIwADAWBgNVHSUBAf8EDDAKBggrBgEFBQcD
# CDBKBgNVHSAEQzBBMDUGDCsGAQQBsjEBAgEDCDAlMCMGCCsGAQUFBwIBFhdodHRw
# czovL3NlY3RpZ28uY29tL0NQUzAIBgZngQwBBAIwSgYDVR0fBEMwQTA/oD2gO4Y5
# aHR0cDovL2NybC5zZWN0aWdvLmNvbS9TZWN0aWdvUHVibGljVGltZVN0YW1waW5n
# Q0FSMzYuY3JsMHoGCCsGAQUFBwEBBG4wbDBFBggrBgEFBQcwAoY5aHR0cDovL2Ny
# dC5zZWN0aWdvLmNvbS9TZWN0aWdvUHVibGljVGltZVN0YW1waW5nQ0FSMzYuY3J0
# MCMGCCsGAQUFBzABhhdodHRwOi8vb2NzcC5zZWN0aWdvLmNvbTANBgkqhkiG9w0B
# AQwFAAOCAYEAAoE+pIZyUSH5ZakuPVKK4eWbzEsTRJOEjbIu6r7vmzXXLpJx4FyG
# mcqnFZoa1dzx3JrUCrdG5b//LfAxOGy9Ph9JtrYChJaVHrusDh9NgYwiGDOhyyJ2
# zRy3+kdqhwtUlLCdNjFjakTSE+hkC9F5ty1uxOoQ2ZkfI5WM4WXA3ZHcNHB4V42z
# i7Jk3ktEnkSdViVxM6rduXW0jmmiu71ZpBFZDh7Kdens+PQXPgMqvzodgQJEkxaI
# ON5XRCoBxAwWwiMm2thPDuZTzWp/gUFzi7izCmEt4pE3Kf0MOt3ccgwn4Kl2FIcQ
# aV55nkjv1gODcHcD9+ZVjYZoyKTVWb4VqMQy/j8Q3aaYd/jOQ66Fhk3NWbg2tYl5
# jhQCuIsE55Vg4N0DUbEWvXJxtxQQaVR5xzhEI+BjJKzh3TQ026JxHhr2fuJ0mV68
# AluFr9qshgwS5SpN5FFtaSEnAwqZv3IS+mlG50rK7W3qXbWwi4hmpylUfygtYLEd
# LQukNEX1jiOKMIIGgjCCBGqgAwIBAgIQNsKwvXwbOuejs902y8l1aDANBgkqhkiG
# 9w0BAQwFADCBiDELMAkGA1UEBhMCVVMxEzARBgNVBAgTCk5ldyBKZXJzZXkxFDAS
# BgNVBAcTC0plcnNleSBDaXR5MR4wHAYDVQQKExVUaGUgVVNFUlRSVVNUIE5ldHdv
# cmsxLjAsBgNVBAMTJVVTRVJUcnVzdCBSU0EgQ2VydGlmaWNhdGlvbiBBdXRob3Jp
# dHkwHhcNMjEwMzIyMDAwMDAwWhcNMzgwMTE4MjM1OTU5WjBXMQswCQYDVQQGEwJH
# QjEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMS4wLAYDVQQDEyVTZWN0aWdvIFB1
# YmxpYyBUaW1lIFN0YW1waW5nIFJvb3QgUjQ2MIICIjANBgkqhkiG9w0BAQEFAAOC
# Ag8AMIICCgKCAgEAiJ3YuUVnnR3d6LkmgZpUVMB8SQWbzFoVD9mUEES0QUCBdxSZ
# qdTkdizICFNeINCSJS+lV1ipnW5ihkQyC0cRLWXUJzodqpnMRs46npiJPHrfLBOi
# fjfhpdXJ2aHHsPHggGsCi7uE0awqKggE/LkYw3sqaBia67h/3awoqNvGqiFRJ+OT
# WYmUCO2GAXsePHi+/JUNAax3kpqstbl3vcTdOGhtKShvZIvjwulRH87rbukNyHGW
# X5tNK/WABKf+Gnoi4cmisS7oSimgHUI0Wn/4elNd40BFdSZ1EwpuddZ+Wr7+Dfo0
# lcHflm/FDDrOJ3rWqauUP8hsokDoI7D/yUVI9DAE/WK3Jl3C4LKwIpn1mNzMyptR
# wsXKrop06m7NUNHdlTDEMovXAIDGAvYynPt5lutv8lZeI5w3MOlCybAZDpK3Dy1M
# Ko+6aEtE9vtiTMzz/o2dYfdP0KWZwZIXbYsTIlg1YIetCpi5s14qiXOpRsKqFKqa
# v9R1R5vj3NgevsAsvxsAnI8Oa5s2oy25qhsoBIGo/zi6GpxFj+mOdh35Xn91y72J
# 4RGOJEoqzEIbW3q0b2iPuWLA911cRxgY5SJYubvjay3nSMbBPPFsyl6mY4/WYucm
# yS9lo3l7jk27MAe145GWxK4O3m3gEFEIkv7kRmefDR7Oe2T1HxAnICQvr9sCAwEA
# AaOCARYwggESMB8GA1UdIwQYMBaAFFN5v1qqK0rPVIDh2JvAnfKyA2bLMB0GA1Ud
# DgQWBBT2d2rdP/0BE/8WoWyCAi/QCj0UJTAOBgNVHQ8BAf8EBAMCAYYwDwYDVR0T
# AQH/BAUwAwEB/zATBgNVHSUEDDAKBggrBgEFBQcDCDARBgNVHSAECjAIMAYGBFUd
# IAAwUAYDVR0fBEkwRzBFoEOgQYY/aHR0cDovL2NybC51c2VydHJ1c3QuY29tL1VT
# RVJUcnVzdFJTQUNlcnRpZmljYXRpb25BdXRob3JpdHkuY3JsMDUGCCsGAQUFBwEB
# BCkwJzAlBggrBgEFBQcwAYYZaHR0cDovL29jc3AudXNlcnRydXN0LmNvbTANBgkq
# hkiG9w0BAQwFAAOCAgEADr5lQe1oRLjlocXUEYfktzsljOt+2sgXke3Y8UPEooU5
# y39rAARaAdAxUeiX1ktLJ3+lgxtoLQhn5cFb3GF2SSZRX8ptQ6IvuD3wz/LNHKpQ
# 5nX8hjsDLRhsyeIiJsms9yAWnvdYOdEMq1W61KE9JlBkB20XBee6JaXx4UBErc+Y
# uoSb1SxVf7nkNtUjPfcxuFtrQdRMRi/fInV/AobE8Gw/8yBMQKKaHt5eia8ybT8Y
# /Ffa6HAJyz9gvEOcF1VWXG8OMeM7Vy7Bs6mSIkYeYtddU1ux1dQLbEGur18ut97w
# gGwDiGinCwKPyFO7ApcmVJOtlw9FVJxw/mL1TbyBns4zOgkaXFnnfzg4qbSvnrwy
# j1NiurMp4pmAWjR+Pb/SIduPnmFzbSN/G8reZCL4fvGlvPFk4Uab/JVCSmj59+/m
# B2Gn6G/UYOy8k60mKcmaAZsEVkhOFuoj4we8CYyaR9vd9PGZKSinaZIkvVjbH/3n
# lLb0a7SBIkiRzfPfS9T+JesylbHa1LtRV9U/7m0q7Ma2CQ/t392ioOssXW7oKLdO
# mMBl14suVFBmbzrt5V5cQPnwtd3UOTpS9oCG+ZZheiIvPgkDmA8FzPsnfXW5qHEL
# B43ET7HHFHeRPRYrMBKjkb8/IN7Po0d0hQoF4TeMM+zYAJzoKQnVKOLg8pZVPT8x
# ggZBMIIGPQIBATBoMFQxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExp
# bWl0ZWQxKzApBgNVBAMTIlNlY3RpZ28gUHVibGljIENvZGUgU2lnbmluZyBDQSBS
# MzYCEDacgYFqXgNIhRHyQAA47UswDQYJYIZIAWUDBAIBBQCggYQwGAYKKwYBBAGC
# NwIBDDEKMAigAoAAoQKAADAZBgkqhkiG9w0BCQMxDAYKKwYBBAGCNwIBBDAcBgor
# BgEEAYI3AgELMQ4wDAYKKwYBBAGCNwIBFTAvBgkqhkiG9w0BCQQxIgQgKrCr1tGQ
# jM9rhhZI/RXTwNDuojX1cDrfnyQ8XYZWvjQwDQYJKoZIhvcNAQEBBQAEggIAwEzM
# dOcbzTl1Jxb/7G/BcHaO+BSaP/0b3Bw33v3snvP1jaU6wOWV880ge/a/zp+ssmy2
# 04n+QJJX92OgRsB0FDLxp6WEieyjzpijUkcE+HWTpRG3yVZswwdrWBBiJNhBx2JO
# gPYXEXYjOXQTBusn2wONy/7qSln3Rzwl5t7M12YWxOTR0Abtj475mhFvg8OfWG3b
# WyxV7KvD+TaTpZbtNAqZtvG+cmXRTpZbY5vR4BgFXXRpgQoUKpI4R4meagjgkUk1
# jJElmDdAFwhlx/XltGz9j9Zx6hEmQ4AJ1QaDNw5wDaYOyAgFVd9ja46HRbUNrHGF
# mPAsCz59H5dKhh7D5v8PjeNU+8WxzcIKjxMVDzMTmt+GQbKdwbPoAzv/5HLA9DQy
# QMUB7f3TcT/9ROzaoXcA00Q9sh8kd36+ZilAN3oTsQuf8BKg+mDW42AhR6++57qD
# Y1pGpuHtEQxVG+ZPKB604PzP1xyIqFoSWTr+++uDdbuXN/1cLvE1rk6Hx+jpmaMW
# 6DVnsUDuVVmYW9qLPvXZ6tfwCC3UAs6ND5DapWiZMM0FwAT7R5hC6/OwFMLRVoZs
# TyOyrSK2eH8H/PXaHDw6vocOzZUfFvKaDNovZvOOJW9u8EsUOYnxSewwvxgL6roF
# QW+CiDnMKuZfrxMefH9AX7VMjY+qkktkzqGQzk6hggMjMIIDHwYJKoZIhvcNAQkG
# MYIDEDCCAwwCAQEwajBVMQswCQYDVQQGEwJHQjEYMBYGA1UEChMPU2VjdGlnbyBM
# aW1pdGVkMSwwKgYDVQQDEyNTZWN0aWdvIFB1YmxpYyBUaW1lIFN0YW1waW5nIENB
# IFIzNgIRAKQpO24e3denNAiHrXpOtyQwDQYJYIZIAWUDBAICBQCgeTAYBgkqhkiG
# 9w0BCQMxCwYJKoZIhvcNAQcBMBwGCSqGSIb3DQEJBTEPFw0yNTExMjQxNDI3MTFa
# MD8GCSqGSIb3DQEJBDEyBDCox8DKt1soppCCbsIogBRIRWW2OWkn8tiJjnDomGkl
# utaTIGfKeAUdRtbUVTF5vM4wDQYJKoZIhvcNAQEBBQAEggIANeiK9GB4ZrW3OA5N
# KBwjB7FZYnJUmDeL22TtU9GXSvdeT8hSGGfY5Qy1nioTt7qNtTcqx8xeoWXyjjen
# h1VfIzMv2dUL/hrDjZlQKODLfqYKFXRgfo7MuWgfZ82OGwHzAkWXMOO3xQVc/ag9
# TGXaVn7/X32M1Aam01h8fhbBFucXjV02JS7yxAuKRPNBhb5ScvPGA7YBniOtrG2l
# RNDct9XfMGrktsmK98KQd9i6FdFJI+NBLlZdSsR0VCb7RGT5nm2OqiL/0s76JJa6
# s6wPRY9iNXJ4xnynjltFCX9F6Fu7gB2PD4FErrETvLJLt3r9caD0QHwxUkwz/AUN
# ElxODO9eKBbzmHzV6DMT7pnva+wFtzUkj8E/ccER8RuSCfY/uMWzHx+N8sg/W7Ac
# E9tYXNNSseMT70C84MXoJZQlFZEpb4fc2Dw/3MmcBTi3TCsa/Z2k3PiVse7kiGk5
# /9y+fFQXmUhG2gTW1XFu/VU2YFx0+jxDNF7LIHEV/yoVMMr174jttr7FMN3hBAjG
# QSNtTa7tG+/IXTGFHcNCT/YySn0bsSKz7B9yYBEPbtoVOPtep4NDh0NlHHLjuGM5
# vMBWhOfVPqzxkDH8WzdS5M6Bpmx8YWxgt3elr5q1jhVLtw3L7aVvZCpktaYKaXF/
# 9xwZ1mGYSmFUNv6O97jFrdvT+r8=
# SIG # End signature block
